<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\Restaurant;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Auth;
use Intervention\Image\Facades\Image;
use Illuminate\Support\Facades\Storage;


class RestaurantController extends Controller
{

    public function showRegisterForm()
    {
        if (!Restaurant::exists()) {
            return view('restaurant.register');
        }elseif (!Auth::check()){
            return redirect()->route('login');
        }
        
    }

    public function register(Request $request)
    {
        $request->validate([
            'name' => 'required|string',
            'phone' => 'required|string',
            'currency' => 'required|string',
            'logo' => 'nullable|image',
            'banner' => 'nullable|image',
            'logo_size' => 'nullable|numeric|min:1|max:100',
            'username' => 'required|string|unique:restaurants',
            'password' => 'required|string|min:3',
        ]);

        $logoPath = null;
        if ($request->hasFile('logo')) {
            $logoPath = $this->saveAsWebp($request->file('logo'), 'logos');
        }

        $bannerPath = null;
        if ($request->hasFile('banner')) {
            $bannerPath = $this->saveAsWebp($request->file('banner'), 'banners');
        }

        Restaurant::create([
            'name' => $request->name,
            'phone' => $request->phone,
            'currency' => $request->currency,
            'logo' => $logoPath,
            'banner' => $bannerPath,
            'logo_size' => $request->logo_size,
            'username' => $request->username,
            'password' => Hash::make($request->password),
        ]);

        return redirect()->route('login')->with('success', 'تم التسجيل بنجاح! سجّل الدخول الآن.');
    }

    /**
     * حفظ الصورة بصيغة WEBP مضغوطة بدون مكتبات خارجية
     */
    private function saveAsWebp($file, $folder)
    {
        $imageInfo = getimagesize($file);
        $extension = strtolower(pathinfo($file->getClientOriginalName(), PATHINFO_EXTENSION));

        // نقرأ الصورة بناءً على نوعها
        switch ($extension) {
            case 'jpeg':
            case 'jpg':
                $image = imagecreatefromjpeg($file);
                break;
            case 'png':
                $image = imagecreatefrompng($file);
                imagepalettetotruecolor($image); // لتحسين جودة PNG
                imagealphablending($image, true);
                imagesavealpha($image, true);
                break;
            case 'gif':
                $image = imagecreatefromgif($file);
                break;
            case 'webp':
                $image = imagecreatefromwebp($file);
                break;
            default:
                return null; // نوع غير مدعوم
        }

        // اسم جديد عشوائي
        $fileName = uniqid() . '.webp';
        $path = storage_path('app/public/' . $folder . '/' . $fileName);

        // نتأكد إن المجلد موجود
        if (!file_exists(dirname($path))) {
            mkdir(dirname($path), 0777, true);
        }

        // نحفظ الصورة بصيغة WEBP وجودة 80%
        imagewebp($image, $path, 80);

        // تنظيف الذاكرة
        imagedestroy($image);

        // نرجع المسار النسبي للتخزين
        return $folder . '/' . $fileName;
    }

}
