<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Restaurant;
use App\Models\Category;
use App\Models\Offer;
use App\Models\Item;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Hash;

class DashboardController extends Controller
{
    function index()
    {
        if (!Restaurant::exists()) {
            return redirect()->route('restaurant.register');
        }

        if (!session()->has('restaurant_id')) {
            return redirect()->route('login');
        }

        $categories = Category::all();
        $offers = Offer::all();
        $items = Item::with('category')->get(); // لازم نربط العلاقة دي في موديل Item

        return view('dashboard.dashboard', compact('categories', 'items', 'offers'));
    }

    public function store(Request $request)
    {
        // التحقق من البيانات
        $request->validate([
            'image' => 'required|image|max:2048',
            'name' => 'required|string|max:255',
            'description' => 'nullable|string|max:255',
            'price_type' => 'required|in:single,multiple',
            'price' => 'nullable|numeric',
            'priceS' => 'nullable|numeric',
            'priceM' => 'nullable|numeric',
            'priceL' => 'nullable|numeric',
            'category_id' => 'nullable|exists:categories,id',
        ]);

        // حفظ الصورة
        $imagePath = null;
        if ($request->hasFile('image')) {
            $imagePath = $this->saveAsWebp($request->file('image'), 'items');
        }

        // معالجة الأسعار حسب نوع السعر
        if ($request->price_type === 'single') {
            // لو سعر واحد فقط
            $priceS = $priceM = $priceL = null;
            $price = $request->price;
        } else {
            // لو أكتر من سعر
            $price = null;
            $priceS = $request->priceS;
            $priceM = $request->priceM;
            $priceL = $request->priceL;
        }

        // إنشاء المنتج
        Item::create([
            'image_url' => $imagePath,
            'name' => $request->name,
            'description' => $request->description,
            'price' => $price,
            'priceS' => $priceS,
            'priceM' => $priceM,
            'priceL' => $priceL,
            'category_id' => $request->category_id,
        ]);

        return redirect()->back()->with('success', '✅ تمت إضافة المنتج بنجاح');
    }


    

    public function categories_store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            
        ]);

        Category::create($request->all());

        return redirect()->back()->with('success', 'تمت الإضافة بنجاح');
    }


    public function Settings()
    {
        if (!session()->has('restaurant_id')) {
            return redirect()->route('login');
        }
        $restaurants = Restaurant::first();
        return view('dashboard.Settings', compact('restaurants'));
    }




    public function Settings_update(Request $request)
    {
        $request->validate([
            'name' => 'required|string',
            'phone' => 'required|string',
            'currency' => 'required|string',
            'logo' => 'nullable|image',
            'banner' => 'nullable|image',
            'logo_size' => 'nullable|numeric|min:1|max:100',
            'username' => 'required|string|unique:restaurants,username,' . Restaurant::first()->id,
            'password' => 'nullable|string|min:3',
        ]);

        $restaurant = Restaurant::first();

        $restaurant->name = $request->name;
        $restaurant->username = $request->username;
        $restaurant->phone = $request->phone;
        $restaurant->currency = $request->currency;
        $restaurant->logo_size = $request->logo_size;

        // تحديث الباسورد لو اتكتب جديد
        if ($request->filled('password')) {
            $restaurant->password = Hash::make($request->password);
        }

        // ✅ تحديث اللوجو وتحويله إلى WebP
        if ($request->hasFile('logo')) {
            if ($restaurant->logo) {
                Storage::disk('public')->delete($restaurant->logo);
            }

            $restaurant->logo = $this->saveAsWebp($request->file('logo'), 'logos');
        }

        // ✅ تحديث البنر وتحويله إلى WebP
        if ($request->hasFile('banner')) {
            if ($restaurant->banner) {
                Storage::disk('public')->delete($restaurant->banner);
            }

            $restaurant->banner = $this->saveAsWebp($request->file('banner'), 'banners');
        }

        $restaurant->save();

        return redirect()->back()->with('success', 'تم تحديث بيانات المطعم بنجاح');
    }

    



    // إظهار النموذج
    public function edit($id)
    {
        if (!session()->has('restaurant_id')) {
            return redirect()->route('login');
        }
        $item = Item::findOrFail($id);
        $categories = Category::all();
        return view('dashboard.edit_items', compact('item', 'categories'));
    }

    public function categories_edit($id)
    {
        if (!session()->has('restaurant_id')) {
            return redirect()->route('login');
        }
        $category = Category::findOrFail($id);
        // return view('dashboard.edit_categories', compact('category'));

        // $categories = Category::all();
        return view('dashboard.edit_categories', compact('category'));
    }

    public function categories_update(Request $request, $id)
    {
        $request->validate([
            'name' => 'required|string|max:255',
        ]);

        $category = Category::findOrFail($id);
        $category->update($request->only(['name']));

        return redirect()->back()->with('success', 'تم تحديث القسم بنجاح');
    }

    // تحديث البيانات
    public function update(Request $request, $id)
    {
        $item = Item::findOrFail($id);
        $request->validate([
            'image' => 'nullable|image|max:2048',
            'name' => 'nullable|string|max:255',
            'description' => 'nullable|string|max:255',
            'price_type' => 'nullable|in:single,multiple',
            'price' => 'nullable|numeric',
            'priceS' => 'nullable|numeric',
            'priceM' => 'nullable|numeric',
            'priceL' => 'nullable|numeric',
            'category_id' => 'nullable|exists:categories,id',
        ]);

        $imagePath = null;
        if ($request->hasFile('image')) {
            if ($item->image_url) {
                Storage::disk('public')->delete($item->image_url);
            }
            $imagePath = $this->saveAsWebp($request->file('image'), 'items');
        }
        if ($request->price_type === 'single') {
            $priceS = $priceM = $priceL = null;
            $price = $request->price;
        } else {
            $price = null;
            $priceS = $request->priceS;
            $priceM = $request->priceM;
            $priceL = $request->priceL;
        }



        $item->update($request->only([
            'name',
            'description',
            'category_id',
        ]) + [
            'image_url' => $imagePath ?? $item->image_url,
            'price' => $price,
            'priceS' => $priceS,
            'priceM' => $priceM,
            'priceL' => $priceL,
        ]);

        return redirect()->back()->with('success', 'تم تحديث المنتج بنجاح');
    }


    public function add_items()
    {
        if (!session()->has('restaurant_id')) {
            return redirect()->route('login');
        }
        $categories = Category::all();
        return view('dashboard.add_items', compact('categories'));
    }

    public function add_categories()
    {
        if (!session()->has('restaurant_id')) {
            return redirect()->route('login');
        }
        return view('dashboard.add_categories');
    }


    public function destroy($id)
    {
        $item = Item::findOrFail($id);
        if ($item->image_url) Storage::disk('public')->delete($item->image_url);
        $item->delete();

        return redirect()->route('dashboard')->with('success', 'تم حذف المنتج بنجاح');
    }

    public function categories_destroy($id)
    {
        $category = Category::findOrFail($id);
        $category->delete();

        return redirect()->route('dashboard')->with('success', 'تم حذف القسم بنجاح');
    }

    public function create_Offer()
    {
        return view('dashboard.add_offer');
    }

    // حفظ عرض جديد
    public function store_Offer(Request $request)
    {
        $request->validate([
            'title' => 'required|string',
            'description' => 'nullable|string',
            'price' => 'required|numeric|min:0',
            'image' => 'nullable|image|max:2048',
            'is_featured' => 'nullable|boolean',
        ]);

        // dd($request->all());

        $imagePath = null;
        if ($request->hasFile('image')) {
            // $imagePath = $request->file('image')->store('offers', 'public');
            $imagePath = $this->saveAsWebp($request->file('image'), 'offers');
        }

        offer::create([
            'title' => $request->title,
            'description' => $request->description,
            'price' => $request->price,
            'image' => $imagePath,
            'is_featured' => $request->is_featured ? true : false,
        ]);

        return redirect()->route('add_offer')->with('success', 'تم إضافة العرض بنجاح ✅');
    }

    // تعديل عرض
    public function edit_Offer(Offer $offer)
    {
        return view('dashboard.edit_Offer', compact('offer'));
    }

    // تحديث عرض
    public function update_Offer(Request $request, Offer $offer)
    {
        $request->validate([
            'title' => 'required|string',
            'description' => 'nullable|string',
            'price' => 'required|numeric|min:0',
            'image' => 'nullable|image|max:2048',
            'is_featured' => 'nullable|boolean',
        ]);

        $imagePath = $offer->image;
        if ($request->hasFile('image')) {
            if ($imagePath) Storage::disk('public')->delete($imagePath);
            $imagePath = $this->saveAsWebp($request->file('image'), 'offers');
        }

        $offer->update([
            'title' => $request->title,
            'description' => $request->description,
            'price' => $request->price,
            'image' => $imagePath,
            'is_featured' => $request->is_featured ? true : false,
        ]);

        return redirect()->back()->with('success', 'تم تحديث العرض بنجاح ✅');
    }

    // حذف عرض
    public function destroy_Offer(Offer $offer)
    {
        if ($offer->image) Storage::disk('public')->delete($offer->image);
        $offer->delete();
        return redirect()->back()->with('success', 'تم حذف العرض بنجاح 🗑️');
    }



    private function saveAsWebp($file, $folder)
    {
        $extension = strtolower($file->getClientOriginalExtension());

        switch ($extension) {
            case 'jpeg':
            case 'jpg':
                $image = imagecreatefromjpeg($file);
                break;
            case 'png':
                $image = imagecreatefrompng($file);
                imagepalettetotruecolor($image);
                imagealphablending($image, true);
                imagesavealpha($image, true);
                break;
            case 'gif':
                $image = imagecreatefromgif($file);
                break;
            case 'webp':
                $image = imagecreatefromwebp($file);
                break;
            default:
                return null; // نوع غير مدعوم
        }

        $fileName = uniqid() . '.webp';
        $path = storage_path('app/public/' . $folder . '/' . $fileName);

        if (!file_exists(dirname($path))) {
            mkdir(dirname($path), 0777, true);
        }

        // نحفظ الصورة بجودة 80%
        imagewebp($image, $path, 80);

        imagedestroy($image);

        return $folder . '/' . $fileName;
    }
}
